import React, {Component} from 'react';
import PropTypes from 'prop-types';
import { Button, Popover, PopoverHeader, PopoverBody, DropdownItem } from 'reactstrap';

import './MainLayout.css'
import Logo from '../../../assets/logo-liferemita.png'
import LogoCorto from '../../../assets/logo-liferemita-corto.png'

import {NavLink} from "react-router-dom";
import classNames from 'classnames';
import {graphql, compose} from "react-apollo";
import {authUserQuery} from "../../../api/queries";
import Ionicon from 'react-ionicons'
import BlinkDot from "../../Loading/BlinkDot";
import Breadcrumbs from "../../Breadcrumbs/Breadcrumbs";
import MonitoraggioSpecieBreadcrumb from "../../Breadcrumbs/MonitoraggioSpecieBreadcrumb";
import MonitoraggioHabitatBreadcrumb from '../../Breadcrumbs/MonitoraggioHabitatBreadcrumb'
import InterventoBreadcrumb from '../../Breadcrumbs/InterventoBreadcrumb'
import DocumentsBreadcrumb from '../../Breadcrumbs/DocumentsBreadcrumb'

import Permissions from '../../../permissions'
import UsersBreadcrumb from "../../Breadcrumbs/UsersBreadcrumb";

const COLOR_GREEN = "#53a044";
const COLOR_PURPLE = "#6d7fcc";
const COLOR_BROWN = "#9b512a";
const COLOR_BLUE = "#6671b2";

const routes = {
  '/': "",
  '/species': "Monitoraggio Specie",
  '/species/create': "Nuova Scheda",
  '/species/:id/edit': "Modifica Scheda",
  '/species/:id': ({id}) => {
    return(<MonitoraggioSpecieBreadcrumb id={id} />);
  },
  '/habitats': "Monitoraggio Habitat",
  '/habitats/create': "Nuova Scheda",
  '/habitats/:id/edit': "Modifica Scheda",
  '/habitats/:id': ({id}) => {
    return(<MonitoraggioHabitatBreadcrumb id={id} />);
  },

  '/interventi': "Interventi",
  '/interventi/create': "Nuovo intervento",
  '/interventi/:id/edit': "Modifica intervento",
  '/interventi/:id': ({id}) => {
    return(<InterventoBreadcrumb id={id} />);
  },

  '/documents': "Documenti",
  '/documents/create': "Nuovo documento",
  '/documents/:id/edit': "Modifica documento",
  '/documents/:id': ({id}) => {
    return(<DocumentsBreadcrumb id={id} />);
  },

  '/users': "Utenti",
  '/users/create': "Nuovo Utente",
  '/users/:id/edit': "Modifica Utente",
  '/users/:id': ({id}) => {
    return(<UsersBreadcrumb id={id} />);
  },
};

const getThemeColor= (theme) => {

  if(!theme) {
    return null;
  }

  switch (theme) {
    case 'purple':
      return COLOR_PURPLE;
    case 'green':
      return COLOR_GREEN;
    case 'brown':
      return COLOR_BROWN;
    case 'blue':
      return COLOR_BLUE;
  }
};

class SidebarMenuItem extends Component {

  static propTypes = {
    to: PropTypes.string.isRequired,
    icon: PropTypes.string.isRequired,
    text: PropTypes.string.isRequired,
    sidebarOpen: PropTypes.bool.isRequired,
    theme: PropTypes.string,
  };

  static defaultProps = {
    theme: "purple"
  };


  constructor(props) {
    super(props);
    this.state = {
      mouseHover: false,
    };

    this.onMouseEnter = this.onMouseEnter.bind(this);
    this.onMouseLeave = this.onMouseLeave.bind(this);
  }

  onMouseEnter() {
    this.setState({mouseHover: true});
  }

  onMouseLeave() {
    this.setState({mouseHover: false});
  }

  render() {
    const {
      to,
      icon,
      text,
      sidebarOpen,
    } = this.props;

    const themeColor = getThemeColor(this.props.theme);

    return(
      <li>
        <NavLink to={to} onMouseLeave={this.onMouseLeave} onMouseEnter={this.onMouseEnter}>
         
	   <Ionicon icon={icon}
	    	fontSize={sidebarOpen ? "1.8em" : "1.5em"}
                   className="nav-icon-link"
                   color={this.state.mouseHover ? themeColor: "#ffffff"}
                    ></Ionicon>  

          <span style={this.state.mouseHover ? {color: themeColor} : null} className="nav-text-link">{text}</span>
        </NavLink>
      </li>
    );
  }
}

class MainLayout extends Component {

  constructor(props) {
    super(props);

    this.state = {
      popoverOpen: false,
      sidebarOpen: false,
    };

    this.toggleUserProfileDropdown = this.toggleUserProfileDropdown.bind(this);
    this.toggleSidebar = this.toggleSidebar.bind(this);
  }

  toggleUserProfileDropdown() {
    this.setState({
      popoverOpen: !this.state.popoverOpen,
    });
  }

  toggleSidebar() {
    this.setState({
      sidebarOpen: !this.state.sidebarOpen,
    });
  }

  render() {
    const {children, data} = this.props;
    const {utenteSessione, loading} = data;

    const themeColor = getThemeColor(this.props.theme);

    return (<div className="wrapper">
      <div id="sidebar" style={{backgroundColor: themeColor}} className={classNames({active: this.state.sidebarOpen})}>
        <div id="sidebar-content">
          <div className="sidebar-header">
            <h3 onClick={() => this.toggleSidebar()}>
              <img className="logo-sidenav" src={Logo} />
            </h3>
            <strong onClick={() => this.toggleSidebar()}><img className="logo-sidenav2" src={LogoCorto} />
            </strong>
          </div>

          <ul className="list-unstyled components">
            <SidebarMenuItem
              theme={this.props.theme}
              to="/species"
              icon="ios-bug"
              text="Monitoraggio Specie"
              sidebarOpen={this.state.sidebarOpen}/>
            <SidebarMenuItem
              theme={this.props.theme}
              to="/habitats"
              icon="ios-leaf"
              text="Monitoraggio Habitat"
              sidebarOpen={this.state.sidebarOpen}/>
            <SidebarMenuItem
              theme={this.props.theme}
              to="/interventi"
              icon="ios-build"
              text="Interventi"
              sidebarOpen={this.state.sidebarOpen}/>
	    {!loading &&
              utenteSessione.ruolo.permessi.filter(item => item.id.endsWith('DOCUMENTO')).length>0 &&
	    <SidebarMenuItem
              theme={this.props.theme}
              to="/documents"
              icon="ios-image"
              text="Documenti"
              sidebarOpen={this.state.sidebarOpen}/>
	    }
              {!loading &&
              utenteSessione.ruolo.permessi.filter(item => item.id == Permissions.adminUsers).length>0 &&
                  <SidebarMenuItem
                      theme={this.props.theme}
                      to="/users"
                      icon="ios-contacts"
                      text="Gestione Utenti"
                      sidebarOpen={this.state.sidebarOpen}/>
              }
          </ul>

        </div>
      </div>

      <div id="content">
        <header className="navbar navbar-expand navbar-light bd-navbar">
          <div className="d-lg-none d-xl-none">
            <a className="navbar-brand" onClick={() => this.toggleSidebar()}>
              <Ionicon icon="ios-menu-outline" fontSize="1.5em" style={{marginRight: '5px', marginLeft: '-8px'}} />
            </a>
          </div>

          <Breadcrumbs routes={routes} />

          <ul className="navbar-nav ml-auto">

            {loading && <li className="nav-item">
              <a className="nav-link text-info">Caricamento in corso <BlinkDot/><BlinkDot/><BlinkDot/></a>
            </li>}
            {(!loading && utenteSessione) && <li className="nav-item dropdown">
              <a id="user-profile-popover" className={classNames(
                "nav-link",
                "dropdown-toggle",
                {"active": this.state.popoverOpen}
              )} onClick={this.toggleUserProfileDropdown}>
                <i className="fa fa-user" aria-hidden="true" />
                {utenteSessione.nome + " " + utenteSessione.cognome}
              </a>
              <Popover style={{zIndex: 9999, padding: '1em 0.2em 0.2em'}} placement="bottom" isOpen={this.state.popoverOpen} target="user-profile-popover" toggle={this.toggleUserProfileDropdown}>
                <PopoverBody>
                  <PopoverHeader>

                  </PopoverHeader>
                  <div className="well well-md well-logout">
                    <small><cite title="San Francisco, USA">Il tuo ruolo utente è:</cite></small>
                    <p>
                      {utenteSessione.ruolo.nomeEsteso}
                    </p>
                    <DropdownItem divider />
                    <a className="btn ionicon-btn btn-info" style={{color: "white"}} onClick={() => window.location.href = "/"}>
                      <Ionicon icon="ios-exit-outline" color="white" fontSize="1.5em" style={{marginRight: '5px', marginLeft: '-8px'}} />
                      Logout
                    </a>
                  </div>
                </PopoverBody>
              </Popover>
            </li>}
          </ul>
        </header>

        {children}
      </div>
    </div>);
  }
}

export default compose(
  graphql(authUserQuery)
)(MainLayout);

MainLayout.defaultProps = {
  theme: "purple"
};

MainLayout.propTypes = {
  theme: PropTypes.string
};
